(function($){
  function escCss(val){
    if (typeof val !== 'string') return '';
    // Keep it permissive, but avoid breaking the style attribute.
    return val.replace(/[^#(),.\-\w\s%/]/g, '');
  }

  function getPresets(){
    return (window.WARSTYLES_ADMIN && window.WARSTYLES_ADMIN.presets) ? window.WARSTYLES_ADMIN.presets : {};
  }

  function setStatus($grid, msg){
    var $s = $grid.find('.warstyles-status');
    if (!$s.length) return;
    $s.stop(true, true)
      .text(msg)
      .css({opacity: 1})
      .delay(1200)
      .animate({opacity: 0}, 400);
  }

  function clearColorInput($input){
    var $wrap = $input.closest('.wp-picker-container');
    var $clear = $wrap.find('.wp-picker-clear');
    if ($clear.length) {
      $clear.trigger('click');
      return;
    }
    $input.val('');
    $input.trigger('change');
  }


  function clamp(n, min, max){
    n = Number(n);
    if (isNaN(n)) n = min;
    if (n < min) n = min;
    if (n > max) n = max;
    return n;
  }

  function parseColor(val){
    val = (val || '').toString().trim();
    if (!val) return null;
    if (val.toLowerCase() === 'transparent') return {r:0,g:0,b:0,a:0};

    // Hex (#rgb or #rrggbb)
    if (val[0] === '#') {
      var hex = val.slice(1);
      if (hex.length === 3) {
        return {
          r: parseInt(hex[0] + hex[0], 16),
          g: parseInt(hex[1] + hex[1], 16),
          b: parseInt(hex[2] + hex[2], 16),
          a: 1
        };
      }
      if (hex.length === 6) {
        return {
          r: parseInt(hex.slice(0,2), 16),
          g: parseInt(hex.slice(2,4), 16),
          b: parseInt(hex.slice(4,6), 16),
          a: 1
        };
      }
      return null;
    }

    // rgb()/rgba()
    var m = val.match(/^rgba?\((.+)\)$/i);
    if (m) {
      var parts = m[1].split(',').map(function(p){ return p.trim(); });
      if (parts.length < 3) return null;

      function to255(p){
        if (p.endsWith('%')) {
          return clamp(Math.round(255 * (parseFloat(p) / 100)), 0, 255);
        }
        return clamp(Math.round(parseFloat(p)), 0, 255);
      }

      var r = to255(parts[0]);
      var g = to255(parts[1]);
      var b = to255(parts[2]);
      var a = 1;
      if (parts.length >= 4) {
        a = parseFloat(parts[3]);
        if (isNaN(a)) a = 1;
        a = clamp(a, 0, 1);
      }
      return {r:r,g:g,b:b,a:a};
    }

    return null;
  }

  function alphaToCss(a){
    a = clamp(a, 0, 1);
    var s = a.toFixed(4).replace(/0+$/, '').replace(/\.$/, '');
    return s === '' ? '0' : s;
  }

  function syncAlphaInputs($grid){
    var pairs = [
      {range:'#warstyles_form_bg_alpha', num:'#warstyles_form_bg_alpha_num'},
      {range:'#warstyles_form_bg2_alpha', num:'#warstyles_form_bg2_alpha_num'}
    ];
    pairs.forEach(function(p){
      var $r = $grid.find(p.range);
      var $n = $grid.find(p.num);
      if (!$r.length || !$n.length) return;

      // Initialize range from number if number is present; else range is already set.
      var nVal = ($n.val() || '').toString().trim();
      if (nVal !== '') {
        $r.val(clamp(parseInt(nVal, 10), 0, 100));
      }

      $r.on('input change', function(){
        $n.val($r.val());
        applyPreview($grid);
      });
      $n.on('input change keyup', function(){
        var v = ($n.val() || '').toString().trim();
        if (v === '') {
          // Optional BG2 can be blank; keep range in sync visually with BG1 if blank.
          if (p.num === '#warstyles_form_bg2_alpha_num') {
            var bg1 = clamp(parseInt(($grid.find('#warstyles_form_bg_alpha_num').val() || '100'), 10), 0, 100);
            $r.val(bg1);
          }
          applyPreview($grid);
          return;
        }
        $r.val(clamp(parseInt(v, 10), 0, 100));
        applyPreview($grid);
      });
    });
  }

  function collectVars($grid){
    var presets = getPresets();
    var presetKey = ($grid.find('#warstyles_preset').val() || 'warchief').toString();
    var preset = presets[presetKey] || {};

    var vars = $.extend({}, (preset.vars || {}));

    var map = {
      '#warstyles_accent': '--wf-accent',
      '#warstyles_form_text': '--wf-form-text',
      '#warstyles_label': '--wf-label',
      '#warstyles_field_bg': '--wf-field-bg',
      '#warstyles_field_text': '--wf-field-text',
      '#warstyles_field_border': '--wf-field-border',
      '#warstyles_placeholder': '--wf-placeholder',
      '#warstyles_form_bg': '--wf-form-bg',
      '#warstyles_form_bg2': '--wf-form-bg2',
      '#warstyles_form_border': '--wf-form-border',
      '#warstyles_button_bg': '--wf-button-bg',
      '#warstyles_button_text': '--wf-button-text',
      '#warstyles_button_border': '--wf-button-border'
    };

    $.each(map, function(sel, varName){
      var $el = $grid.find(sel);
      if (!$el.length) return;
      var val = ($el.val() || '').toString().trim();
      if (val !== '') {
        vars[varName] = val;
      }
    });

    return { presetKey: presetKey, vars: vars };
  }

  function applyPreview($grid){
    var $preview = $grid.find('#warstyles-preview-form');
    if (!$preview.length) return;

    var enabled = $grid.find('input[name="warstyles_enabled"]').is(':checked');
    var bgMode = ($grid.find('#warstyles_bg_mode').val() === 'gradient') ? 'gradient' : 'solid';
    var stretch = $grid.find('#warstyles_stretch').is(':checked');

    var data = collectVars($grid);

    var style = '';
    if (data.vars) {
      Object.keys(data.vars).forEach(function(k){
        var v = data.vars[k];
        if (v === null || typeof v === 'undefined') return;
        v = String(v).trim();
        if (!v) return;
        style += k + ':' + escCss(v) + ';';
      });
    }


    // Background alpha controls (multiply any preset rgba alpha)
    var pct1 = clamp(parseInt(($grid.find('#warstyles_form_bg_alpha_num').val() || '100'), 10), 0, 100);
    var pct2raw = ($grid.find('#warstyles_form_bg2_alpha_num').val() || '').toString().trim();
    var pct2 = (pct2raw === '') ? pct1 : clamp(parseInt(pct2raw, 10), 0, 100);

    var c1 = (data.vars && data.vars['--wf-form-bg']) ? data.vars['--wf-form-bg'] : 'transparent';
    var c2 = (data.vars && data.vars['--wf-form-bg2']) ? data.vars['--wf-form-bg2'] : c1;

    var p1 = parseColor(c1);
    var p2 = parseColor(c2);

    if (p1) {
      style += '--wf-form-bg-rgb:' + p1.r + ',' + p1.g + ',' + p1.b + ';';
      style += '--wf-form-bg-alpha:' + alphaToCss(p1.a * (pct1 / 100)) + ';';
    }
    if (p2) {
      style += '--wf-form-bg2-rgb:' + p2.r + ',' + p2.g + ',' + p2.b + ';';
      style += '--wf-form-bg2-alpha:' + alphaToCss(p2.a * (pct2 / 100)) + ';';
    }

    $preview.attr('style', style);

    // Preset class swap.
    $preview.removeClass(function(i, cls){
      var m = cls.match(/\bwarstyles-preset-[^\s]+/g);
      return m ? m.join(' ') : '';
    });
    $preview.addClass('warstyles-preset-' + data.presetKey);

    $preview.toggleClass('warstyles-bg-gradient', bgMode === 'gradient');
    $preview.toggleClass('warstyles-stretch', !!stretch);
    $preview.toggleClass('is-disabled', !enabled);
  }


  function initSubtabs($grid){
    var $tabs = $grid.find('.warstyles-subtabs .nav-tab');
    if (!$tabs.length) return;

    var formId = $grid.data('warstyles-form-id') || '';
    var storageKey = 'warstyles_active_view_' + formId;

    function setView(view){
      if (!view) view = 'controls';
      if (!$grid.find('.warstyles-subview-' + view).length) view = 'controls';

      $grid.removeClass('warstyles-view-controls warstyles-view-preview warstyles-view-css')
           .addClass('warstyles-view-' + view);

      $tabs.removeClass('nav-tab-active');
      $tabs.filter('[data-warstyles-view="' + view + '"]').addClass('nav-tab-active');

      applyPreview($grid);
    }

    // Restore last view (if any) after the controls initialize.
    var initial = 'controls';
    try {
      if (window.localStorage) {
        initial = localStorage.getItem(storageKey) || 'controls';
      }
    } catch (e) {}
    setView(initial);

    $tabs.on('click', function(e){
      e.preventDefault();
      var view = ($(this).data('warstyles-view') || '').toString();
      if (!view) return;
      setView(view);
      try {
        if (window.localStorage) localStorage.setItem(storageKey, view);
      } catch (e) {}
    });
  }



  function initGrid($grid){
    // Init WP color picker.
    $grid.find('.warstyles-color').each(function(){
      var $input = $(this);
      if ($input.data('wpColorPicker')) return;
      $input.wpColorPicker({
        change: function(){ applyPreview($grid); },
        clear: function(){ applyPreview($grid); }
      });
    });

    // Alpha range/number sync.
    syncAlphaInputs($grid);

    // Live update.
    $grid.on('change keyup', 'input.warstyles-color, #warstyles_bg_mode, #warstyles_stretch, input[name="warstyles_enabled"], #warstyles_form_bg_alpha, #warstyles_form_bg_alpha_num, #warstyles_form_bg2_alpha, #warstyles_form_bg2_alpha_num', function(){
      applyPreview($grid);
    });

    // Selecting a preset should override any existing selections.
    $grid.on('change', '#warstyles_preset', function(){
      $grid.find('.warstyles-color').each(function(){
        clearColorInput($(this));
      });
      applyPreview($grid);
      setStatus($grid, 'Preset applied (overrides cleared)');
    });

    // Manual reset button.
    $grid.on('click', '.warstyles-reset-overrides', function(e){
      e.preventDefault();
      $grid.find('.warstyles-color').each(function(){
        clearColorInput($(this));
      });
      applyPreview($grid);
      setStatus($grid, 'Overrides cleared');
    });

    // Initial render.
    applyPreview($grid);
    initSubtabs($grid);
  }

  $(function(){
    $('.warstyles-builder-grid').each(function(){
      initGrid($(this));
    });
  });
})(jQuery);
