<?php
if (!defined('ABSPATH')) exit;

add_action('admin_post_warforms_submit', 'warforms_reboot_handle_submit');
add_action('admin_post_nopriv_warforms_submit', 'warforms_reboot_handle_submit');

function warforms_reboot_handle_submit(): void {
    $form_id = isset($_POST['wf_form_id']) ? (int)$_POST['wf_form_id'] : 0;
    if ($form_id <= 0) {
        warforms_reboot_set_last_error('form', 'missing form id');
        warforms_reboot_safe_redirect_back(['wf_token' => warforms_reboot_make_flash(['status' => 'fail'])]);
    }

    $post = get_post($form_id);
    if (!$post || $post->post_type !== 'warform') {
        warforms_reboot_set_last_error('form', 'invalid form id');
        warforms_reboot_safe_redirect_back(['wf_token' => warforms_reboot_make_flash(['status' => 'fail'])]);
    }

    // Nonce
    $nonce = isset($_POST['wf_nonce']) ? (string)wp_unslash($_POST['wf_nonce']) : '';
    if (!$nonce || !wp_verify_nonce($nonce, 'warforms_submit_' . $form_id)) {
        warforms_reboot_set_last_error('nonce', 'nonce failed');
        $token = warforms_reboot_make_flash(['status' => 'fail', 'errors' => ['_form' => 'Security check failed. Please refresh and try again.']]);
        warforms_reboot_safe_redirect_back(['wf_token' => $token]);
    }

    // Honeypot
    $hp = isset($_POST['wf_hp']) ? (string)wp_unslash($_POST['wf_hp']) : '';
    if (trim($hp) !== '') {
        warforms_reboot_set_last_error('honeypot', 'honeypot tripped');
        $token = warforms_reboot_make_flash(['status' => 'fail', 'errors' => ['_form' => 'Submission rejected.']]);
        warforms_reboot_safe_redirect_back(['wf_token' => $token]);
    }

    $schema = warforms_reboot_get_schema($form_id);
    $submitted = isset($_POST['wf']) && is_array($_POST['wf']) ? (array)wp_unslash($_POST['wf']) : [];

    // Phone field UX: if segmented inputs were used, merge them into wf[field] before validation.
    $phone_parts = isset($_POST['wf_phone']) && is_array($_POST['wf_phone']) ? (array) wp_unslash($_POST['wf_phone']) : [];
    if (!empty($phone_parts)) {
        foreach ($phone_parts as $fname => $parts) {
            if (!is_string($fname) || $fname === '' || !is_array($parts)) continue;
            $current = $submitted[$fname] ?? '';
            if (is_array($current)) continue;
            if (trim((string)$current) !== '') continue; // hidden value already provided
            $area  = preg_replace('/\D+/', '', (string)($parts['area'] ?? ''));
            $pref  = preg_replace('/\D+/', '', (string)($parts['prefix'] ?? ''));
            $line  = preg_replace('/\D+/', '', (string)($parts['line'] ?? ''));
            $digits = $area . $pref . $line;
            if ($digits !== '') {
                $submitted[$fname] = $digits;
            }
        }
    }


    // Currency field UX: accept legacy wf_currency[field][amount|currency] payloads and merge into wf[field] before validation.
    $currency_parts = isset($_POST['wf_currency']) && is_array($_POST['wf_currency']) ? (array) wp_unslash($_POST['wf_currency']) : [];
    if (!empty($currency_parts)) {
        foreach ($currency_parts as $fname => $parts) {
            if (!is_string($fname) || $fname === '' || !is_array($parts)) continue;
            $current = $submitted[$fname] ?? '';
            if (is_array($current)) continue;
            if (is_string($current) && trim($current) !== '') continue; // hidden value already provided
            $submitted[$fname] = $parts;
        }
    }




    $result = warforms_reboot_validate($schema, $submitted);

    if (!empty($result['ok'])) {
        $values = is_array($result['values'] ?? null) ? (array)$result['values'] : [];

        // Step 8: persist entry on success
        $context = [
            'user_id' => get_current_user_id(),
            'ip' => isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field((string)$_SERVER['REMOTE_ADDR']) : '',
            'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field((string)$_SERVER['HTTP_USER_AGENT']) : '',
        ];

        $entry_id = 0;
        if (function_exists('warforms_reboot_insert_entry')) {
            $inserted = warforms_reboot_insert_entry($form_id, $values, $context);

            if (is_wp_error($inserted)) {
                $detail = '';
                $data = $inserted->get_error_data();
                if (is_array($data) && isset($data['wpdb_last_error'])) {
                    $detail = (string)$data['wpdb_last_error'];
                }
                if ($detail === '') {
                    $detail = $inserted->get_error_message();
                }

                warforms_reboot_set_last_error('db', $detail);

                $token = warforms_reboot_make_flash([
                    'status' => 'fail',
                    'errors' => ['_form' => 'We could not save your submission. Please try again.'],
                    'values' => $values,
                ]);
                warforms_reboot_safe_redirect_back(['wf_token' => $token]);
            }

            $entry_id = (int)$inserted;
            do_action('warforms_entry_created', $entry_id, $form_id, $values);
        }

        warforms_reboot_set_last_error('ok', '');

// Step 10: Honor confirmation on success.
$conf = isset($schema['confirmation']) && is_array($schema['confirmation']) ? $schema['confirmation'] : [];
$conf_type = isset($conf['type']) ? (string) $conf['type'] : '';
if ($conf_type === 'redirect') {
    $url = isset($conf['redirect_url']) ? (string) $conf['redirect_url'] : '';
    $url = esc_url_raw($url);
    if ($url !== '') {
        if ($entry_id > 0) {
            $url = add_query_arg(['wf_entry' => $entry_id], $url);
        }
        wp_safe_redirect($url);
        exit;
    }
}

        $token = warforms_reboot_make_flash([
            'status' => 'success',
            'entry_id' => $entry_id,
        ]);
        warforms_reboot_safe_redirect_back(['wf_token' => $token]);
    }

    // fail
    warforms_reboot_set_last_error('validation', 'validation failed');
    $token = warforms_reboot_make_flash([
        'status' => 'fail',
        'errors' => $result['errors'] ?? [],
        'values' => $result['values'] ?? [],
    ]);
    warforms_reboot_safe_redirect_back(['wf_token' => $token]);
}

/**
 * Create a short-lived flash transient and return token.
 */
function warforms_reboot_make_flash(array $data): string {
    $token = wp_generate_password(20, false, false);
    $data = array_merge([
        'form_id' => isset($_POST['wf_form_id']) ? (int)$_POST['wf_form_id'] : 0,
        'status' => 'fail',
        'errors' => [],
        'values' => [],
    ], $data);

    set_transient('warforms_reboot_flash_' . $token, $data, 10 * MINUTE_IN_SECONDS);
    return $token;
}